// ******************************************************************
// VISPack. Copyright (c) 1994-2000 Ross Whitaker rtw@utk.edu       *
// For conditions of distribution and use, see the file LICENSE.txt *
// accompanying this distribution.                                  *
// ******************************************************************

// $Id: mathutil.h,v 1.3 2003/04/30 01:32:13 whitaker Exp $

#ifndef	MATH_MATHUTIL_H
#define	MATH_MATHUTIL_H

#include <stdlib.h>
#include <limits.h>
#include <float.h>
#include <math.h>
#include "util/defs.h"

// ---------------------------------------------------------------------------

const VISReal half_pi = M_PI / 2.0;
const VISReal one_and_a_half_pi = 1.5 * M_PI;
const VISReal two_pi = 2.0 * M_PI;


float gaussFast(float);
float gaussCumFast(float);
float cosFast(float);
float acosFast(float);

// ---------------------------------------------------------------------------
// min, max, and abs functions -- done "correctly" as an inline (i.e. typed) function

template <class T> 
inline T tabs(const T& a)
{
    return (a < (T)0 ? (-a) : a);
}

template <class T> 
inline T tsign(const T& a)
{
    return (a < (T)0 ? (-1) : 1);
}

//returns a^n
template <class T>
T power(T a, int n)
{
    T total = 1.0;
    int i;
    if (n == 0) return(1);
    if (n >= 0)
	for (i = 0; i < n; i++)
	    {
		total *= a;
	    }
    else
	for (i = 0; i < -n; i++)
	    {
		total /= a;
	    }
    return(total);
}

/*{
    float total = 1.0f;
    int i;
    if (n >= 0)
	for (i = 0; i < n; i++)
	    {
		total *= a;
	    }
    else
	for (i = 0; i < n; i++)
	    {
		total /= a;
	    }
    return(total);
    }*/

// This needs to be changed 3-14-00 to prevent clashes with std functions 
// by the same name for the same purpose -  RTW

// CHANGED 6/28/00 to VISMin to avoid STL clashes. --JC
template <class T> inline const T& VISmin(const T& a, const T& b)
{
    return (a < b ? a : b);
}

template <class T> inline const T& VISmin(const T& a,const T& b,const T& c)
{
    return VISmin(a,VISmin(b,c));
}

template <class T> inline const T& VISmin(const T& a,const T& b,
				       const T& c,const T& d)
{
    return VISmin(VISmin(a,b),VISmin(c,d));
}

template <class T> inline const T& VISmin(const T& a,const T& b,
				       const T& c,const T& d,
				       const T& e,const T& f)
{
    return VISmin(VISmin(a,b,c,d),VISmin(e,f));
}

// ---------------------------------------------------------------------------
// max function -- done "correctly" as an inline (i.e. typed) function

// CHANGED 6/28/00 to VISMax to avoid STL clashes. --JC

template <class T> inline const T& VISmax(const T& a,const T& b)
{
    return (a > b ? a : b);
}

template <class T> inline const T& VISmax(const T& a,const T& b,const T& c)
{
    return VISmax(a,VISmax(b,c));
}

template <class T> inline const T& VISMaxAbs(const T& a,const T& b)
{
    return (tabs(a) > tabs(b) ? a : b);
}

template <class T> inline const T& VISmax(const T& a,const T& b,
				       const T& c,const T& d)
{
    return VISmax(VISmax(a,b),VISmax(c,d));
}

template <class T> inline const T& VISmax(const T& a,const T& b,
				       const T& c,const T& d,
				       const T& e,const T& f)
{
    return VISmax(VISmax(a,b,c,d),VISmax(e,f));
}

// ---------------------------------------------------------------------------

inline VISReal rad_to_deg(VISReal radians)
{
    return (radians * 180.0 / M_PI);
}

// ---------------------------------------------------------------------------

inline VISReal deg_to_rad(VISReal degrees)
{
    return (degrees * M_PI / 180.0);
}


// ---------------------------------------------------------------------------

// returns a uniformly distributed deviate on the 0-1.0 interval
float rand1();

// returns a normally distributed deviate with zero mean an unit variance, 
// using ran1() as the source of uniform deviates
float gasdev();
// returns a MEAN=1 gamma dist of order ia                                           
float gamdev(int ia);                                                                


// ---------------------------------------------------------------------------
// Simple square routine - inline so should work ok
// Its templated so you can use any type you want -- VISReal or double or
// float or int.

template <class T> inline T sqr(T x) { return x*x; }

// ---------------------------------------------------------------------------

template <class T> inline T cub(T x) { return x*x*x; }

// ---------------------------------------------------------------------------

#define DELTA_GAUSS (0.01f)
const float _GAUSS_TABLE[512]
=
{
1.0,
0.99995,
0.99980002,
0.9995501,
0.99920032,
0.99875078,
0.99820162,
0.997553,
0.99680511,
0.99595819,
0.99501248,
0.99396826,
0.99282586,
0.9915856,
0.99024786,
0.98881304,
0.98728157,
0.9856539,
0.98393051,
0.98211193,
0.98019867,
0.97819132,
0.97609047,
0.97389674,
0.97161077,
0.96923323,
0.96676484,
0.9642063,
0.96155838,
0.95882184,
0.95599748,
0.95308613,
0.95008863,
0.94700586,
0.9438387,
0.94058806,
0.9372549,
0.93384015,
0.93034481,
0.92676987,
0.92311635,
0.91938529,
0.91557774,
0.91169479,
0.90773754,
0.90370708,
0.89960455,
0.8954311,
0.89118789,
0.88687609,
0.8824969,
0.87805153,
0.87354119,
0.86896711,
0.86433055,
0.85963276,
0.85487502,
0.85005859,
0.84518478,
0.84025488,
0.83527021,
0.83023208,
0.82514182,
0.82000077,
0.81481026,
0.80957165,
0.80428628,
0.79895552,
0.79358073,
0.78816328,
0.78270454,
0.77720588,
0.77166867,
0.76609431,
0.76048416,
0.7548396,
0.74916202,
0.7434528,
0.73771331,
0.73194493,
0.72614904,
0.720327,
0.7144802,
0.70860998,
0.70271772,
0.69680478,
0.69087249,
0.68492222,
0.67895529,
0.67297305,
0.66697681,
0.6609679,
0.65494763,
0.6489173,
0.6428782,
0.63683161,
0.63077882,
0.62472108,
0.61865965,
0.61259576,
0.60653066,
0.60046555,
0.59440166,
0.58834016,
0.58228224,
0.57622907,
0.57018181,
0.5641416,
0.55810956,
0.5520868,
0.54607443,
0.54007352,
0.53408515,
0.52811036,
0.5221502,
0.51620567,
0.5102778,
0.50436756,
0.49847592,
0.49260384,
0.48675226,
0.48092209,
0.47511424,
0.46932959,
0.46356902,
0.45783336,
0.45212346,
0.44644012,
0.44078414,
0.4351563,
0.42955736,
0.42398805,
0.41844911,
0.41294123,
0.4074651,
0.40202138,
0.39661073,
0.39123378,
0.38589113,
0.38058338,
0.3753111,
0.37007485,
0.36487516,
0.35971256,
0.35458755,
0.3495006,
0.34445218,
0.33944274,
0.33447271,
0.32954248,
0.32465247,
0.31980303,
0.31499453,
0.31022731,
0.30550168,
0.30081795,
0.29617642,
0.29157734,
0.28702097,
0.28250755,
0.2780373,
0.27361042,
0.2692271,
0.26488752,
0.26059182,
0.25634015,
0.25213264,
0.24796938,
0.24385049,
0.23977603,
0.23574608,
0.23176068,
0.22781987,
0.22392368,
0.22007211,
0.21626517,
0.21250282,
0.20878506,
0.20511182,
0.20148305,
0.1978987,
0.19435867,
0.19086288,
0.18741123,
0.18400359,
0.18063985,
0.17731987,
0.1740435,
0.17081059,
0.16762097,
0.16447446,
0.16137087,
0.15831002,
0.1552917,
0.15231569,
0.14938178,
0.14648972,
0.1436393,
0.14083025,
0.13806233,
0.13533528,
0.13264883,
0.13000271,
0.12739663,
0.12483031,
0.12230345,
0.11981577,
0.11736694,
0.11495667,
0.11258464,
0.11025053,
0.10795401,
0.10569476,
0.10347244,
0.10128672,
0.09913725,
0.0970237,
0.09494571,
0.09290294,
0.09089503,
0.08892162,
0.08698235,
0.08507688,
0.08320482,
0.08136582,
0.07955951,
0.07778552,
0.07604348,
0.07433302,
0.07265377,
0.07100535,
0.0693874,
0.06779953,
0.06624138,
0.06471256,
0.0632127,
0.06174144,
0.06029838,
0.05888317,
0.05749542,
0.05613476,
0.05480083,
0.05349324,
0.05221163,
0.05095563,
0.04972487,
0.04851899,
0.04733762,
0.04618039,
0.04504695,
0.04393693,
0.04284998,
0.04178575,
0.04074387,
0.03972399,
0.03872577,
0.03774886,
0.03679291,
0.03585759,
0.03494255,
0.03404745,
0.03317197,
0.03231577,
0.03147852,
0.03065989,
0.02985957,
0.02907723,
0.02831255,
0.02756523,
0.02683495,
0.02612141,
0.0254243,
0.02474331,
0.02407816,
0.02342855,
0.02279418,
0.02217477,
0.02157004,
0.0209797,
0.02040348,
0.01984109,
0.01929229,
0.01875678,
0.01823432,
0.01772463,
0.01722747,
0.01674258,
0.01626971,
0.01580862,
0.01535906,
0.01492079,
0.01449357,
0.01407718,
0.01367139,
0.01327597,
0.01289069,
0.01251534,
0.01214971,
0.01179358,
0.01144674,
0.011109,
0.01078014,
0.01045997,
0.01014829,
0.00984492,
0.00954966,
0.00926233,
0.00898274,
0.00871073,
0.0084461,
0.0081887,
0.00793835,
0.00769488,
0.00745813,
0.00722795,
0.00700417,
0.00678664,
0.0065752,
0.00636972,
0.00617004,
0.00597602,
0.00578753,
0.00560442,
0.00542656,
0.00525382,
0.00508607,
0.00492318,
0.00476504,
0.00461151,
0.00446248,
0.00431784,
0.00417747,
0.00404125,
0.00390909,
0.00378087,
0.0036565,
0.00353586,
0.00341885,
0.00330539,
0.00319538,
0.00308872,
0.00298532,
0.00288509,
0.00278795,
0.00269381,
0.00260259,
0.0025142,
0.00242858,
0.00234563,
0.0022653,
0.00218749,
0.00211215,
0.0020392,
0.00196857,
0.00190019,
0.00183401,
0.00176996,
0.00170797,
0.00164799,
0.00158995,
0.00153381,
0.0014795,
0.00142697,
0.00137617,
0.00132705,
0.00127955,
0.00123362,
0.00118923,
0.00114632,
0.00110485,
0.00106477,
0.00102604,
0.00098862,
0.00095247,
0.00091755,
0.00088383,
0.00085125,
0.0008198,
0.00078943,
0.00076011,
0.0007318,
0.00070448,
0.00067811,
0.00065266,
0.00062811,
0.00060441,
0.00058156,
0.00055951,
0.00053824,
0.00051773,
0.00049796,
0.00047889,
0.0004605,
0.00044277,
0.00042569,
0.00040922,
0.00039335,
0.00037806,
0.00036333,
0.00034914,
0.00033546,
0.00032229,
0.00030961,
0.00029739,
0.00028563,
0.00027431,
0.00026341,
0.00025292,
0.00024282,
0.0002331,
0.00022375,
0.00021475,
0.00020609,
0.00019776,
0.00018975,
0.00018205,
0.00017464,
0.00016751,
0.00016066,
0.00015408,
0.00014775,
0.00014166,
0.00013582,
0.0001302,
0.0001248,
0.00011961,
0.00011463,
0.00010984,
0.00010525,
0.00010083,
9.6593E-05,
9.2523E-05,
8.8616E-05,
8.4865E-05,
8.1265E-05,
7.7809E-05,
7.4493E-05,
7.1312E-05,
6.8259E-05,
6.5331E-05,
6.2522E-05,
5.9827E-05,
5.7243E-05,
5.4766E-05,
5.239E-05,
5.0112E-05,
4.7929E-05,
4.5836E-05,
4.383E-05,
4.1907E-05,
4.0065E-05,
3.83E-05,
3.661E-05,
3.499E-05,
3.3439E-05,
3.1953E-05,
3.053E-05,
2.9168E-05,
2.7863E-05,
2.6615E-05,
2.5419E-05,
2.4275E-05,
2.318E-05,
2.2133E-05,
2.113E-05,
2.0171E-05,
1.9254E-05,
1.8376E-05,
1.7537E-05,
1.6734E-05,
1.5967E-05,
1.5233E-05,
1.4531E-05,
1.3861E-05,
1.322E-05,
1.2607E-05,
1.2022E-05,
1.1462E-05,
1.0928E-05,
1.0417E-05,
9.9295E-06,
9.4637E-06,
9.0188E-06,
8.594E-06,
8.1883E-06,
7.8011E-06,
7.4314E-06,
7.0785E-06,
6.7417E-06,
6.4203E-06,
6.1136E-06,
5.8209E-06,
5.5418E-06,
5.2754E-06,
5.0214E-06,
4.7791E-06,
4.5481E-06,
4.3278E-06,
4.1178E-06,
3.9175E-06,
3.7267E-06,
3.5447E-06,
3.3713E-06,
3.2061E-06,
3.0487E-06,
2.8987E-06,
2.7558E-06,
2.6197E-06,
2.4901E-06,
2.3666E-06,
2.2491E-06,
0.0
};

const float _GAUSSCUM_TABLE[512]
=
{
0.5,
0.50398937,
0.50797834,
0.51196651,
0.51595348,
0.51993886,
0.52392225,
0.52790325,
0.53188147,
0.5358565,
0.53982796,
0.54379544,
0.54775857,
0.55171694,
0.55567017,
0.559617,
0.563559,
0.56749514,
0.57142393,
0.57534566,
0.57925995,
0.58316642,
0.58706469,
0.5909544,
0.59483517,
0.59870663,
0.60256843,
0.60642021,
0.6102616,
0.61409224,
0.6179118,
0.62171991,
0.62551624,
0.62930044,
0.63307217,
0.6368311,
0.6405769,
0.64430923,
0.64802779,
0.65173224,
0.65542227,
0.65909757,
0.66275783,
0.66640275,
0.67003204,
0.67364539,
0.67724251,
0.68082313,
0.68438696,
0.68793373,
0.69146315,
0.69497498,
0.69846894,
0.70194478,
0.70540225,
0.7088411,
0.71226108,
0.71566197,
0.71904353,
0.72240554,
0.72574776,
0.72907,
0.73237203,
0.73565365,
0.73891466,
0.74215487,
0.74537409,
0.74857213,
0.75174882,
0.75490397,
0.75803744,
0.76114904,
0.76423864,
0.76730606,
0.77035118,
0.77337385,
0.77637393,
0.7793513,
0.78230583,
0.78523741,
0.78814592,
0.79103126,
0.79389331,
0.796732,
0.79954722,
0.8023389,
0.80510695,
0.80785129,
0.81057186,
0.8132686,
0.81594144,
0.81859034,
0.82121524,
0.8238161,
0.82639289,
0.82894557,
0.83147412,
0.83397851,
0.83645872,
0.83891475,
0.84134658,
0.84375422,
0.84613766,
0.84849691,
0.85083199,
0.85314292,
0.8554297,
0.85769237,
0.85993097,
0.86214551,
0.86433605,
0.86650263,
0.86864529,
0.87076409,
0.87285908,
0.87493032,
0.87697788,
0.87900183,
0.88100223,
0.88297918,
0.88493273,
0.88686298,
0.88877002,
0.89065393,
0.89251482,
0.89435277,
0.89616789,
0.89796029,
0.89973007,
0.90147733,
0.90320221,
0.9049048,
0.90658524,
0.90824364,
0.90988014,
0.91149485,
0.9130879,
0.91465944,
0.9162096,
0.91773851,
0.91924632,
0.92073317,
0.9221992,
0.92364456,
0.9250694,
0.92647387,
0.92785812,
0.92922231,
0.93056659,
0.93189112,
0.93319607,
0.93448158,
0.93574784,
0.93699499,
0.9382232,
0.93943265,
0.94062349,
0.94179591,
0.94295006,
0.94408611,
0.94520425,
0.94630464,
0.94738746,
0.94845287,
0.94950107,
0.95053221,
0.95154648,
0.95254405,
0.9535251,
0.9544898,
0.95543834,
0.95637089,
0.95728763,
0.95818874,
0.9590744,
0.95994477,
0.96080005,
0.96164041,
0.96246602,
0.96327707,
0.96407373,
0.96485618,
0.96562459,
0.96637915,
0.96712002,
0.96784739,
0.96856143,
0.9692623,
0.96995019,
0.97062528,
0.97128772,
0.97193769,
0.97257537,
0.97320092,
0.97381452,
0.97441632,
0.97500651,
0.97558524,
0.97615268,
0.976709,
0.97725435,
0.97778891,
0.97831283,
0.97882627,
0.9793294,
0.97982236,
0.98030533,
0.98077844,
0.98124187,
0.98169575,
0.98214025,
0.98257551,
0.98300168,
0.98341891,
0.98382735,
0.98422715,
0.98461843,
0.98500136,
0.98537607,
0.9857427,
0.98610139,
0.98645227,
0.98679548,
0.98713116,
0.98745943,
0.98778043,
0.9880943,
0.98840114,
0.9887011,
0.9889943,
0.98928087,
0.98956091,
0.98983456,
0.99010194,
0.99036316,
0.99061833,
0.99086759,
0.99111102,
0.99134876,
0.9915809,
0.99180756,
0.99202885,
0.99224487,
0.99245572,
0.99266151,
0.99286234,
0.99305831,
0.99324952,
0.99343607,
0.99361804,
0.99379554,
0.99396866,
0.99413748,
0.99430211,
0.99446262,
0.99461911,
0.99477165,
0.99492034,
0.99506526,
0.99520649,
0.99534411,
0.99547819,
0.99560882,
0.99573607,
0.99586002,
0.99598074,
0.99609831,
0.99621278,
0.99632424,
0.99643276,
0.99653839,
0.99664121,
0.99674128,
0.99683867,
0.99693343,
0.99702563,
0.99711534,
0.9972026,
0.99728747,
0.99737002,
0.9974503,
0.99752836,
0.99760425,
0.99767804,
0.99774977,
0.99781949,
0.99788725,
0.9979531,
0.99801709,
0.99807926,
0.99813966,
0.99819834,
0.99825533,
0.99831068,
0.99836443,
0.99841663,
0.9984673,
0.9985165,
0.99856427,
0.99861062,
0.99865562,
0.99869928,
0.99874165,
0.99878276,
0.99882264,
0.99886132,
0.99889885,
0.99893524,
0.99897054,
0.99900476,
0.99903794,
0.99907011,
0.99910129,
0.99913152,
0.99916082,
0.99918921,
0.99921671,
0.99924337,
0.99926919,
0.9992942,
0.99931843,
0.9993419,
0.99936462,
0.99938662,
0.99940793,
0.99942855,
0.99944852,
0.99946784,
0.99948655,
0.99950465,
0.99952216,
0.99953911,
0.9995555,
0.99957136,
0.9995867,
0.99960154,
0.99961588,
0.99962976,
0.99964317,
0.99965614,
0.99966867,
0.99968079,
0.9996925,
0.99970381,
0.99971475,
0.99972531,
0.99973552,
0.99974538,
0.9997549,
0.9997641,
0.99977298,
0.99978156,
0.99978984,
0.99979783,
0.99980555,
0.999813,
0.99982019,
0.99982713,
0.99983382,
0.99984028,
0.99984651,
0.99985252,
0.99985832,
0.99986391,
0.9998693,
0.9998745,
0.99987952,
0.99988435,
0.99988901,
0.9998935,
0.99989783,
0.999902,
0.99990602,
0.99990989,
0.99991362,
0.99991721,
0.99992067,
0.99992401,
0.99992722,
0.99993031,
0.99993328,
0.99993615,
0.9999389,
0.99994156,
0.99994411,
0.99994657,
0.99994894,
0.99995121,
0.9999534,
0.99995551,
0.99995754,
0.99995949,
0.99996136,
0.99996316,
0.99996489,
0.99996656,
0.99996816,
0.9999697,
0.99997118,
0.9999726,
0.99997396,
0.99997528,
0.99997654,
0.99997775,
0.99997891,
0.99998003,
0.9999811,
0.99998213,
0.99998312,
0.99998407,
0.99998498,
0.99998585,
0.99998669,
0.9999875,
0.99998827,
0.99998901,
0.99998973,
0.99999041,
0.99999106,
0.99999169,
0.99999229,
0.99999287,
0.99999342,
0.99999395,
0.99999446,
0.99999495,
0.99999542,
0.99999586,
0.99999629,
0.9999967,
0.9999971,
0.99999748,
0.99999784,
0.99999818,
0.99999851,
0.99999883,
0.99999914,
0.99999943,
0.99999971,
0.99999997,
1.00000023,
1.00000047,
1.0000007,
1.00000093,
1.00000114,
1.00000135,
1.00000154,
1.00000173,
1.00000191,
1.00000208,
1.00000224,
1.0000024,
1.00000255,
1.00000269,
1.00000283,
1.00000296,
1.00000308,
1.0000032,
1.00000331,
1.00000342,
1.00000353,
1.00000363,
1.00000372,
1.00000381,
1.0000039,
1.00000398,
1.00000406,
1.00000413,
1.00000421,
1.00000427,
1.00000434,
1.0000044,
1.00000446,
1.00000452,
1.00000457,
1.00000462,
1.00000467,
1.00000472,
1.00000476,
1.00000481,
1.00000485,
1.00000489,
1.00000492,
1.00000496,
1.00000499,
1.00000502,
1.00000505,
1.00000508,
1.00000511,
1.00000514,
1.00000516,
1.00000518,
1.00000521,
1.00000523,
1.00000525,
1.00000527,
1.00000529,
1.00000531,
1.00000532,
1.00000534,
1.00000535,
1.00000537,
1.00000538,
1.00000539,
1.00000541,
1.00000542,
1.00000543,
1.00000544,
1.00000545,
1.00000546,
1.00000547,
1.00000548
};


#define DELTA_COS (0.003074f)
const float _COS_TABLE[512]
=
{
1,
0.99999528,
0.9999811,
0.99995748,
0.99992441,
0.99988188,
0.99982991,
0.9997685,
0.99969763,
0.99961732,
0.99952756,
0.99942836,
0.99931971,
0.99920163,
0.99907409,
0.99893712,
0.99879071,
0.99863486,
0.99846958,
0.99829485,
0.9981107,
0.99791711,
0.9977141,
0.99750165,
0.99727979,
0.99704849,
0.99680778,
0.99655764,
0.99629809,
0.99602913,
0.99575075,
0.99546296,
0.99516577,
0.99485917,
0.99454317,
0.99421778,
0.99388299,
0.9935388,
0.99318523,
0.99282228,
0.99244994,
0.99206822,
0.99167713,
0.99127667,
0.99086684,
0.99044765,
0.9900191,
0.9895812,
0.98913394,
0.98867734,
0.98821139,
0.98773611,
0.98725149,
0.98675754,
0.98625427,
0.98574168,
0.98521977,
0.98468856,
0.98414804,
0.98359822,
0.98303911,
0.9824707,
0.98189302,
0.98130605,
0.98070981,
0.98010431,
0.97948954,
0.97886552,
0.97823225,
0.97758973,
0.97693798,
0.97627699,
0.97560678,
0.97492735,
0.97423871,
0.97354086,
0.97283381,
0.97211757,
0.97139215,
0.97065754,
0.96991376,
0.96916082,
0.96839872,
0.96762747,
0.96684707,
0.96605754,
0.96525888,
0.9644511,
0.9636342,
0.9628082,
0.9619731,
0.96112891,
0.96027564,
0.9594133,
0.95854189,
0.95766142,
0.9567719,
0.95587334,
0.95496575,
0.95404913,
0.9531235,
0.95218887,
0.95124523,
0.95029261,
0.949331,
0.94836043,
0.94738089,
0.9463924,
0.94539497,
0.94438861,
0.94337332,
0.94234912,
0.94131601,
0.94027401,
0.93922312,
0.93816336,
0.93709473,
0.93601725,
0.93493092,
0.93383576,
0.93273177,
0.93161897,
0.93049737,
0.92936697,
0.92822779,
0.92707984,
0.92592313,
0.92475767,
0.92358347,
0.92240055,
0.92120891,
0.92000856,
0.91879952,
0.9175818,
0.9163554,
0.91512035,
0.91387665,
0.91262432,
0.91136336,
0.91009379,
0.90881562,
0.90752886,
0.90623352,
0.90492963,
0.90361718,
0.90229619,
0.90096668,
0.89962865,
0.89828212,
0.8969271,
0.89556361,
0.89419166,
0.89281125,
0.89142241,
0.89002515,
0.88861947,
0.8872054,
0.88578295,
0.88435212,
0.88291294,
0.88146541,
0.88000956,
0.87854539,
0.87707292,
0.87559216,
0.87410312,
0.87260583,
0.87110029,
0.86958652,
0.86806453,
0.86653434,
0.86499596,
0.86344941,
0.8618947,
0.86033184,
0.85876086,
0.85718176,
0.85559456,
0.85399927,
0.85239592,
0.85078451,
0.84916506,
0.84753758,
0.8459021,
0.84425862,
0.84260717,
0.84094775,
0.83928039,
0.8376051,
0.83592189,
0.83423078,
0.83253179,
0.83082494,
0.82911023,
0.82738769,
0.82565733,
0.82391916,
0.82217322,
0.8204195,
0.81865803,
0.81688882,
0.8151119,
0.81332727,
0.81153496,
0.80973498,
0.80792734,
0.80611207,
0.80428919,
0.8024587,
0.80062064,
0.798775,
0.79692182,
0.79506111,
0.79319289,
0.79131717,
0.78943397,
0.78754331,
0.78564521,
0.78373969,
0.78182676,
0.77990644,
0.77797876,
0.77604372,
0.77410135,
0.77215166,
0.77019468,
0.76823042,
0.7662589,
0.76428014,
0.76229416,
0.76030097,
0.7583006,
0.75629306,
0.75427838,
0.75225657,
0.75022765,
0.74819165,
0.74614857,
0.74409844,
0.74204128,
0.73997711,
0.73790595,
0.73582781,
0.73374272,
0.7316507,
0.72955176,
0.72744593,
0.72533322,
0.72321366,
0.72108727,
0.71895406,
0.71681406,
0.71466729,
0.71251376,
0.7103535,
0.70818653,
0.70601287,
0.70383253,
0.70164554,
0.69945193,
0.6972517,
0.69504488,
0.6928315,
0.69061157,
0.68838511,
0.68615215,
0.68391271,
0.6816668,
0.67941445,
0.67715569,
0.67489052,
0.67261897,
0.67034107,
0.66805684,
0.66576629,
0.66346945,
0.66116634,
0.65885699,
0.65654141,
0.65421962,
0.65189165,
0.64955752,
0.64721726,
0.64487088,
0.6425184,
0.64015986,
0.63779526,
0.63542464,
0.63304801,
0.6306654,
0.62827683,
0.62588233,
0.62348191,
0.62107559,
0.61866341,
0.61624539,
0.61382154,
0.61139189,
0.60895646,
0.60651528,
0.60406837,
0.60161575,
0.59915744,
0.59669347,
0.59422387,
0.59174865,
0.58926784,
0.58678145,
0.58428953,
0.58179208,
0.57928914,
0.57678072,
0.57426685,
0.57174755,
0.56922286,
0.56669278,
0.56415735,
0.56161659,
0.55907052,
0.55651916,
0.55396255,
0.55140071,
0.54883365,
0.54626141,
0.543684,
0.54110146,
0.5385138,
0.53592106,
0.53332325,
0.5307204,
0.52811254,
0.52549969,
0.52288187,
0.52025911,
0.51763143,
0.51499887,
0.51236143,
0.50971916,
0.50707207,
0.50442018,
0.50176353,
0.49910214,
0.49643603,
0.49376523,
0.49108977,
0.48840966,
0.48572494,
0.48303563,
0.48034176,
0.47764335,
0.47494042,
0.472233,
0.46952113,
0.46680481,
0.46408409,
0.46135898,
0.45862951,
0.4558957,
0.45315759,
0.4504152,
0.44766855,
0.44491767,
0.44216258,
0.43940332,
0.43663991,
0.43387237,
0.43110072,
0.42832501,
0.42554525,
0.42276147,
0.41997369,
0.41718194,
0.41438625,
0.41158665,
0.40878315,
0.4059758,
0.40316461,
0.4003496,
0.39753082,
0.39470828,
0.391882,
0.38905203,
0.38621838,
0.38338108,
0.38054016,
0.37769564,
0.37484755,
0.37199592,
0.36914077,
0.36628214,
0.36342004,
0.36055451,
0.35768558,
0.35481326,
0.35193759,
0.3490586,
0.34617631,
0.34329074,
0.34040193,
0.33750991,
0.3346147,
0.33171632,
0.32881481,
0.32591019,
0.3230025,
0.32009175,
0.31717797,
0.3142612,
0.31134146,
0.30841878,
0.30549318,
0.3025647,
0.29963335,
0.29669918,
0.2937622,
0.29082245,
0.28787994,
0.28493472,
0.28198681,
0.27903623,
0.27608301,
0.27312719,
0.27016878,
0.26720782,
0.26424434,
0.26127836,
0.25830991,
0.25533902,
0.25236571,
0.24939002,
0.24641198,
0.24343161,
0.24044893,
0.23746399,
0.2344768,
0.23148739,
0.2284958,
0.22550205,
0.22250617,
0.21950818,
0.21650812,
0.21350601,
0.21050189,
0.20749578,
0.20448771,
0.2014777,
0.19846579,
0.19545201,
0.19243638,
0.18941893,
0.18639969,
0.18337869,
0.18035595,
0.17733151,
0.1743054,
0.17127764,
0.16824826,
0.16521729,
0.16218476,
0.1591507,
0.15611513,
0.15307809,
0.1500396,
0.14699969,
0.1439584,
0.14091574,
0.13787176,
0.13482647,
0.1317799,
0.12873209,
0.12568307,
0.12263285,
0.11958148,
0.11652898,
0.11347537,
0.1104207,
0.10736498,
0.10430825,
0.10125053,
0.09819185,
0.09513225,
0.09207174,
0.08901037,
0.08594816,
0.08288513,
0.07982132,
0.07675676,
0.07369147,
0.07062548,
0.06755883,
0.06449154,
0.06142364,
0.05835516,
0.05528612,
0.05221657,
0.04914652,
0.04607601,
0.04300506,
0.03993371,
0.03686197,
0.03378989,
0.03071749,
0.0276448,
0.02457185,
0.02149867,
0.01842528,
0.01535172,
0.01227802,
0.0092042,
0.00613029,
0.00305632,
0.00
};

#define DELTA_ACOS (0.00195695f)
const float _ACOS_TABLE[512]
=
{
0,
0.06257133,
0.08850367,
0.10841213,
0.12520401,
0.14000525,
0.15339317,
0.16571071,
0.17718125,
0.18795994,
0.1981597,
0.2078658,
0.21714446,
0.22604831,
0.23462005,
0.24289485,
0.2509021,
0.25866665,
0.26620975,
0.27354972,
0.28070252,
0.2876821,
0.29450079,
0.30116953,
0.30769808,
0.31409517,
0.3203687,
0.32652579,
0.33257291,
0.33851597,
0.34436036,
0.35011102,
0.35577251,
0.36134903,
0.36684445,
0.37226238,
0.37760615,
0.38287888,
0.38808346,
0.39322261,
0.39829887,
0.40331461,
0.40827208,
0.41317337,
0.41802047,
0.42281525,
0.42755946,
0.43225479,
0.43690281,
0.44150503,
0.44606285,
0.45057765,
0.4550507,
0.45948322,
0.46387639,
0.46823132,
0.47254906,
0.47683064,
0.48107702,
0.48528913,
0.48946786,
0.49361407,
0.49772858,
0.50181215,
0.50586556,
0.50988952,
0.51388474,
0.51785187,
0.52179157,
0.52570445,
0.52959112,
0.53345216,
0.53728811,
0.54109953,
0.54488692,
0.54865079,
0.55239162,
0.55610989,
0.55980604,
0.56348053,
0.56713377,
0.57076617,
0.57437813,
0.57797005,
0.5815423,
0.58509524,
0.58862922,
0.5921446,
0.5956417,
0.59912085,
0.60258236,
0.60602654,
0.60945369,
0.61286409,
0.61625804,
0.61963579,
0.62299763,
0.62634381,
0.62967459,
0.63299021,
0.63629092,
0.63957694,
0.64284851,
0.64610585,
0.64934918,
0.65257872,
0.65579466,
0.65899722,
0.66218659,
0.66536297,
0.66852654,
0.67167749,
0.674816,
0.67794225,
0.68105641,
0.68415864,
0.68724912,
0.69032801,
0.69339546,
0.69645163,
0.69949668,
0.70253074,
0.70555397,
0.7085665,
0.71156849,
0.71456005,
0.71754134,
0.72051248,
0.72347359,
0.72642482,
0.72936627,
0.73229807,
0.73522035,
0.73813321,
0.74103677,
0.74393115,
0.74681645,
0.74969279,
0.75256027,
0.755419,
0.75826907,
0.7611106,
0.76394367,
0.76676839,
0.76958485,
0.77239315,
0.77519337,
0.77798562,
0.78076997,
0.78354653,
0.78631536,
0.78907657,
0.79183022,
0.79457642,
0.79731522,
0.80004673,
0.802771,
0.80548813,
0.80819818,
0.81090123,
0.81359736,
0.81628664,
0.81896913,
0.82164491,
0.82431405,
0.82697662,
0.82963268,
0.8322823,
0.83492554,
0.83756247,
0.84019315,
0.84281765,
0.84543602,
0.84804833,
0.85065463,
0.85325499,
0.85584945,
0.85843809,
0.86102096,
0.8635981,
0.86616958,
0.86873545,
0.87129576,
0.87385057,
0.87639993,
0.87894389,
0.8814825,
0.8840158,
0.88654386,
0.88906671,
0.89158441,
0.894097,
0.89660453,
0.89910704,
0.90160459,
0.90409721,
0.90658495,
0.90906786,
0.91154598,
0.91401934,
0.916488,
0.91895199,
0.92141136,
0.92386614,
0.92631638,
0.92876212,
0.93120339,
0.93364024,
0.9360727,
0.93850081,
0.94092461,
0.94334414,
0.94575942,
0.94817051,
0.95057742,
0.95298021,
0.9553789,
0.95777353,
0.96016414,
0.96255075,
0.9649334,
0.96731212,
0.96968695,
0.97205792,
0.97442506,
0.97678841,
0.97914798,
0.98150383,
0.98385596,
0.98620443,
0.98854924,
0.99089045,
0.99322807,
0.99556213,
0.99789267,
1.00021971,
1.00254327,
1.0048634,
1.00718011,
1.00949344,
1.0118034,
1.01411003,
1.01641336,
1.0187134,
1.02101019,
1.02330375,
1.02559411,
1.0278813,
1.03016533,
1.03244623,
1.03472403,
1.03699875,
1.03927042,
1.04153906,
1.04380469,
1.04606734,
1.04832703,
1.05058378,
1.05283762,
1.05508857,
1.05733664,
1.05958188,
1.06182429,
1.06406389,
1.06630072,
1.06853478,
1.07076611,
1.07299472,
1.07522064,
1.07744388,
1.07966446,
1.08188242,
1.08409775,
1.0863105,
1.08852067,
1.09072828,
1.09293336,
1.09513593,
1.097336,
1.09953359,
1.10172872,
1.10392142,
1.10611169,
1.10829956,
1.11048505,
1.11266817,
1.11484895,
1.11702739,
1.11920352,
1.12137736,
1.12354892,
1.12571822,
1.12788528,
1.13005012,
1.13221274,
1.13437318,
1.13653144,
1.13868754,
1.1408415,
1.14299333,
1.14514305,
1.14729068,
1.14943624,
1.15157973,
1.15372118,
1.1558606,
1.157998,
1.1601334,
1.16226682,
1.16439828,
1.16652777,
1.16865533,
1.17078097,
1.1729047,
1.17502653,
1.17714649,
1.17926458,
1.18138082,
1.18349522,
1.1856078,
1.18771857,
1.18982755,
1.19193475,
1.19404019,
1.19614387,
1.19824581,
1.20034602,
1.20244453,
1.20454134,
1.20663646,
1.20872991,
1.2108217,
1.21291184,
1.21500036,
1.21708725,
1.21917253,
1.22125623,
1.22333834,
1.22541888,
1.22749787,
1.22957531,
1.23165122,
1.23372561,
1.23579849,
1.23786988,
1.23993979,
1.24200823,
1.2440752,
1.24614074,
1.24820483,
1.2502675,
1.25232877,
1.25438863,
1.2564471,
1.2585042,
1.26055993,
1.26261431,
1.26466734,
1.26671904,
1.26876943,
1.2708185,
1.27286628,
1.27491277,
1.27695798,
1.27900193,
1.28104463,
1.28308608,
1.2851263,
1.28716529,
1.28920308,
1.29123966,
1.29327506,
1.29530928,
1.29734232,
1.29937421,
1.30140495,
1.30343455,
1.30546303,
1.30749038,
1.30951663,
1.31154178,
1.31356585,
1.31558884,
1.31761076,
1.31963162,
1.32165144,
1.32367022,
1.32568797,
1.3277047,
1.32972042,
1.33173515,
1.33374889,
1.33576165,
1.33777344,
1.33978426,
1.34179414,
1.34380308,
1.34581108,
1.34781817,
1.34982434,
1.3518296,
1.35383397,
1.35583746,
1.35784007,
1.35984182,
1.3618427,
1.36384274,
1.36584194,
1.3678403,
1.36983785,
1.37183458,
1.37383051,
1.37582565,
1.37782,
1.37981357,
1.38180637,
1.38379842,
1.38578971,
1.38778027,
1.38977009,
1.39175918,
1.39374756,
1.39573523,
1.39772221,
1.39970849,
1.40169409,
1.40367902,
1.40566329,
1.4076469,
1.40962986,
1.41161218,
1.41359387,
1.41557494,
1.4175554,
1.41953525,
1.4215145,
1.42349316,
1.42547125,
1.42744876,
1.4294257,
1.43140209,
1.43337794,
1.43535324,
1.43732801,
1.43930226,
1.44127599,
1.44324921,
1.44522194,
1.44719417,
1.44916593,
1.4511372,
1.45310801,
1.45507836,
1.45704826,
1.45901772,
1.46098674,
1.46295533,
1.4649235,
1.46689126,
1.46885862,
1.47082558,
1.47279216,
1.47475835,
1.47672417,
1.47868963,
1.48065473,
1.48261948,
1.48458389,
1.48654797,
1.48851172,
1.49047516,
1.49243828,
1.4944011,
1.49636362,
1.49832586,
1.50028782,
1.50224951,
1.50421093,
1.5061721,
1.50813302,
1.51009369,
1.51205413,
1.51401435,
1.51597435,
1.51793414,
1.51989372,
1.52185311,
1.52381231,
1.52577133,
1.52773018,
1.52968886,
1.53164738,
1.53360576,
1.53556399,
1.53752209,
1.53948005,
1.5414379,
1.54339564,
1.54535327,
1.5473108,
1.54926825,
1.55122561,
1.55318289,
1.55514011,
1.55709727,
1.55905437,
1.56101143,
1.56296846,
1.56492545,
1.56688242,
1.56883938,
1.57079633
};



#endif
